unit threadedICYServer;

{* Delphi Streaming Radio Library
 * Copyright 2004-2007, Steve Blinch
 * http://code.blitzaffe.com
 * ============================================================================
 *
 * LICENSE
 *
 * This code is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *}

interface

uses
  IdTCPServer,IdTCPConnection,Classes,IdDateTimeStamp,rawbufferUnit,threadLogUnit,threadedICYClient;

const
  MSG_ICYSERVER_FETCH_BUFFER    = 1701;
  MSG_ICYSERVER_CONNECTED       = 1702;
  MSG_ICYSERVER_DISCONNECTED    = 1703;

  MIN_STARTUP_BLOCKS     = 10;

Type
  TICYServerThread = class(TIdPeerThread)
    private
      BytesSent: Integer;
      MetaBytes: Integer;

      MetaFirst: Boolean;
      StartingUp: Boolean;
    public
      OutboundBlocks: TList;
      OutboundPending: Boolean;
      Kill: Boolean;

      ThreadLog: TThreadLog;

      MetaData: String;
      MetaInterval: Integer;
      MetaChanged: Boolean;
      MetaRequested: Boolean;

      constructor Create(ACreateSuspended: Boolean = True); override;
      destructor Destroy; override;

      procedure MetaWriteBuffer(Buffer: TRawBuffer);

      // called from VCL thread 
      procedure QueueBuffer(Buffer: TRawBuffer);
      procedure SetMetaData(NewMetaData: String);

    end;
  TThreadedICYServer = class(TIdTCPServer)
    private
      FWindowHandle: THandle;
      FMetaData: String;

      MetaInterval: Integer;

      procedure threadExecute(AThread: TIdPeerThread);
      procedure HTTPError(Connection: TIdTCPConnection; Time: String; Code: Integer);
    public

      StreamInfo: TStreamInfo;

      constructor Create(AOwner: TComponent); override;

      procedure StreamData(Buffer: TRawBuffer);
      procedure SetMetaData(NewMetaData: String);
      procedure Shutdown;

      property MetaData: String read FMetaData;
    end;

implementation

uses IdException,SysUtils,Windows,Messages,Forms;

constructor TThreadedICYServer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FMetaData:='';
  MetaInterval:=16000;

  FillChar(StreamInfo,SizeOf(StreamInfo),0);
  StreamInfo.Bitrate:=128;

  ThreadClass:=TICYServerThread;

  if (AOwner is TForm) then
    FWindowHandle:=(AOwner as TForm).Handle;

  OnExecute:=threadExecute;
end;

procedure SplitRequest(GETCommand: String; var HTTPMethod,HTTPURI,HTTPVersion: String);
var p: Integer;
begin
  p:=Pos(' ',GETCommand);
  HTTPMethod:=Copy(GETCommand,1,p-1);
  Delete(GETCommand,1,p);

  p:=Pos(' ',GETCommand);
  HTTPURI:=Copy(GETCommand,1,p-1);
  Delete(GETCommand,1,p);

  HTTPVersion:=GETCommand;
  p:=pos('/',HTTPVersion);
  Delete(HTTPVersion,1,p);
//GET /path/file.html HTTP/1.1
end;

procedure TThreadedICYServer.HTTPError(Connection: TIdTCPConnection; Time: String; Code: Integer);
Type
  THTTPError = record
    Code: Integer;
    Summary: String;
    Content: String;
  end;
const
  MAX_CODES = 3;
  Codes: Array[0..MAX_CODES-1] of THTTPError = (
    (Code: 400; Summary: 'Bad request'; Content: 'Your browser sent a request that this server could not understand.'),
    (Code: 404; Summary: 'Not found'; Content: 'The requested URL was not found on this server.'),
    (Code: 500; Summary: 'Internal server error'; Content: 'The server encountered an internal error or misconfiguration and was unable to complete your request.')
  );
var
  CodeIdx: Integer;
begin
  for CodeIdx:=0 to MAX_CODES-1 do
    if Codes[CodeIdx].Code=Code then break;
  if CodeIdx>MAX_CODES-1 then CodeIdx:=MAX_CODES-1;

  Connection.WriteLn(Format('HTTP/1.0 %d %s',[Code,Codes[CodeIdx].Summary]));
  Connection.WriteLn(Format('Date: %s',[Time]));
  Connection.WriteLn('Content-type: text/html');
  Connection.WriteLn(Format('Status: %d %s',[Code,Codes[CodeIdx].Summary]));
  Connection.WriteLn('<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML 2.0//EN">');
  Connection.WriteLn('<HTML><HEAD>');
  Connection.WriteLn(Format('<TITLE>%d %s</TITLE>',[Code,Codes[CodeIdx].Summary]));
  Connection.WriteLn('</HEAD><BODY>');
  Connection.WriteLn(Format('<H1>%s</H1>',[Codes[CodeIdx].Summary]));
  Connection.WriteLn(Format('%s.<P>',[Codes[CodeIdx].Content]));
  Connection.WriteLn('<HR>');
//  Connection.WriteLn('<ADDRESS>Apache/1.3.29 Server at 63.247.81.147 Port 80</ADDRESS>');
end;

procedure TThreadedICYServer.threadExecute(AThread: TIdPeerThread);
var
  ICYThread: TICYServerThread;
  ThreadLog: TThreadLog;
  Host: String;
  S,GETCommand: String;
  HTTPVersion,HTTPURI,HTTPMethod: String;
  DateTime: TIdDateTimeStamp;

  OutboundBuffer: TRawBuffer;
  OutboundReady: Boolean;
begin
  SendMessage(FWindowHandle,WM_USER,MSG_ICYSERVER_CONNECTED,Integer(AThread));

  ThreadLog:=TThreadLog.Create(FWindowHandle,AThread);
  DateTime:=TIdDateTimeStamp.Create(nil);
  DateTime.SetFromTDateTime(Now);
  try

    Host:=AThread.Connection.Socket.Binding.PeerIP;
    ThreadLog.Log('ICY server thread connected ('+Host+')',LL_VERBOSE);

    ICYThread:=(AThread as TICYServerThread);

    ICYThread.MetaInterval:=MetaInterval;
    ICYThread.ThreadLog:=ThreadLog;
    ICYThread.SetMetaData(FMetaData);
    ICYThread.MetaRequested:=False;


    try
      S:='';
      while (AThread.Connection.Connected) and (not AThread.Terminated) do
        begin
          S:=AThread.Connection.ReadLn;
          if GETCommand='' then GETCommand:=S;
          if S='' then break;
          if Copy(LowerCase(S),1,13)='icy-metadata:' then ICYThread.MetaRequested:=True;

        end;
    except
      on E: Exception do
        begin
          ThreadLog.Log('ICY server caught exception: '+E.Message,LL_NORMAL);
          exit;
        end;
    end;
    SplitRequest(GETCommand,HTTPMethod,HTTPURI,HTTPVersion);

    if (HTTPVersion<>'1.0') then
      begin HTTPError(AThread.Connection,DateTime.AsRFC822,400); exit; end;
    if (HTTPMethod<>'GET') then
      begin HTTPError(AThread.Connection,DateTime.AsRFC822,400); exit; end;

    with AThread.Connection do
      begin
        WriteLn('HTTP/'+HTTPVersion+' 200 OK');
        WriteLn('Content-Type: audio/mpeg');

// All of the following headers need to be passed through from the peer stream
// when we get the peer stream subsystem going

        WriteLn('icy-metaint: '+IntToStr(MetaInterval));

        WriteLn('icy-br: '+IntToStr(StreamInfo.BitRate));
        if Length(StreamInfo.Desc)>0 then WriteLn('icy-description: '+StreamInfo.Desc);
        if Length(StreamInfo.Genre)>0 then WriteLn('icy-genre: '+StreamInfo.Genre);
        if Length(StreamInfo.Name)>0 then
          WriteLn('icy-name: '+StreamInfo.Name+' (via threadedICYServer)')
        else
          WriteLn('icy-name: threadedICYServer stream relay');
        if Length(StreamInfo.Pub)>0 then WriteLn('icy-pub: 0');
        if Length(StreamInfo.URL)>0 then WriteLn('icy-url: '+StreamInfo.URL);
        WriteLn('Server: threadedICYServer 1.0');
        WriteLn('');
      end;

    while (not ICYThread.Kill) and (not AThread.Terminated) do
      begin
        AThread.Connection.ReadFromStack(True,1,False);
        if not AThread.Connection.Connected then break;

        if (AThread.Connection.InputBuffer.Size>0) then
          AThread.Connection.ReadString(AThread.Connection.InputBuffer.Size); // discard any input

        OutboundReady:=ICYThread.OutboundPending and not ( (ICYThread.StartingUp) and (ICYThread.OutboundBlocks.Count<MIN_STARTUP_BLOCKS) );

        if (OutboundReady) then
          begin
//            ThreadLog.Log('HTTP::threadExecute() Outbound data pending',LL_DEBUG);

            OutboundBuffer:=TRawBuffer.Create(ICYThread);

            try
              {$IFDEF RAWBUF_DEBUG}OutboundBuffer.SetID('TThreadedICYServer.threadExecute.OutboundBuffer');{$ENDIF}

              SendMessage(FWindowHandle,WM_USER,MSG_ICYSERVER_FETCH_BUFFER,Integer(OutboundBuffer));

              if OutboundBuffer.Size=0 then
                ThreadLog.Log('HTTP::threadExecute() WTF?  Fetched ICY buffer size is 0!',LL_NORMAL)
              else
                //ThreadLog.Log(IntToStr(OutboundBuffer.Size),LL_NORMAL)
                ;

              ICYThread.MetaWriteBuffer(OutboundBuffer);

              //AThread.Connection.WriteBuffer(OutboundBuffer.Buffer^,OutboundBuffer.Size);
              if not AThread.Connection.Connected then break;

            finally
              OutboundBuffer.Free;
            end;

          end;
      end;

  finally

    try
      AThread.Connection.Disconnect;
    except
      on E: EIdConnClosedGracefully do
        ; // ignore EIdConnClosedGracefully
      on E: Exception do
        ThreadLog.Log('Disconnect threw exception: '+E.Message,LL_NORMAL);
    end;

    ThreadLog.Log('ICY server thread disconnected ('+Host+')',LL_VERBOSE);
    ThreadLog.Free;
    DateTime.Free;

    SendMessage(FWindowHandle,WM_USER,MSG_ICYSERVER_DISCONNECTED,Integer(AThread));

    AThread.Terminate;
  end;

end;

procedure TThreadedICYServer.StreamData(Buffer: TRawBuffer);
var
  i: Integer;
  StreamThread: TThread;
begin

  with Threads.LockList do
    try
      for i:=0 to Count-1 do
        begin
          StreamThread:=Items[i];

          (StreamThread as TICYServerThread).QueueBuffer(Buffer);
        end;
    finally
      Threads.UnlockList;
    end;

end;

procedure TThreadedICYServer.SetMetaData(NewMetaData: String);
var
  i: Integer;
  StreamThread: TThread;
begin
  // cache this for new clients
  FMetaData:=NewMetaData;

  // then notify all existing clients
  with Threads.LockList do
    try
      for i:=0 to Count-1 do
        begin
          StreamThread:=Items[i];

          (StreamThread as TICYServerThread).SetMetaData(FMetaData);
        end;
    finally
      Threads.UnlockList;
    end;

end;

procedure TThreadedICYServer.Shutdown;
var
  i: Integer;
  StreamThread: TThread;
begin
  with Threads.LockList do
    try
      for i:=0 to Count-1 do
        begin
          StreamThread:=Items[i];
          (StreamThread as TICYServerThread).Kill:=True;
        end;
    finally
      Threads.UnlockList;
    end;
end;

constructor TICYServerThread.Create(ACreateSuspended: Boolean = True);
begin
  inherited Create(ACreateSuspended);

  BytesSent:=0;
  MetaBytes:=0;
  MetaFirst:=True;
  MetaChanged:=False;
  StartingUp:=True;

  Kill:=False;
  OutboundBlocks:=TList.Create;
  OutboundPending:=False;
end;

destructor TICYServerThread.Destroy;
var i: Integer;
begin
  for i:=0 to OutboundBlocks.Count-1 do
    TRawBuffer(OutboundBlocks[i]).Free;

  OutboundBlocks.Free;
  inherited;
end;

// Makes a copy of Buffer and adds it to the outbound blocks list for this thread
procedure TICYServerThread.QueueBuffer(Buffer: TRawBuffer);
var ThreadBuffer: TRawBuffer;
begin
  ThreadBuffer:=Buffer.Copy;

  OutboundBlocks.Add(ThreadBuffer);
  OutboundPending:=True;

  if (StartingUp) and (OutboundBlocks.Count>=MIN_STARTUP_BLOCKS) then StartingUp:=False;
end;

procedure TICYServerThread.SetMetaData(NewMetaData: String);
begin
  MetaData:=NewMetaData;
  MetaChanged:=True;
end;

procedure TICYServerThread.MetaWriteBuffer(Buffer: TRawBuffer);
var
  BytesBefore: Integer;
  BytesAfter: Integer;
  MetaString: String;
  MetaLength: Integer;
  MetaByte: Byte;
  RestOfBuffer: Pointer;
begin

  if (MetaRequested) and (MetaBytes+Buffer.Size>=MetaInterval) then
    begin
      // calculate out where our boundaries are
      BytesBefore:=MetaInterval-MetaBytes;
      BytesAfter:=Buffer.Size-BytesBefore;
{
      ThreadLog.Log(
        Format('MetaBytes: %d; Buffer.Size: %d; MetaInterval: %d; BytesBefore: %d; BytesAfter: %d',[MetaBytes,Buffer.Size,MetaInterval,BytesBefore,BytesAfter]),
        LL_NORMAL
      );
}
      if (MetaChanged or MetaFirst) then
        begin
          MetaChanged:=False;
          MetaFirst:=False;
          MetaString:=MetaData;

          // make sure the metadata isn't more than 255*16 characters (or more specifically,
          // 254*16 + the length of "StreamTitle='';")
          if (Length(MetaString)>254*16) then SetLength(MetaString,254*16);
          MetaString:='StreamTitle='''+MetaString+''';';

          // calculate our desired length
          MetaLength:=Length(MetaString)+15;
          if (MetaLength mod 16 <> 0) then MetaLength:=MetaLength+16-(MetaLength mod 16);

          // pad the string out to the desired length, and add a byte at the beginning to
          // hold the length byte
          MetaString:=' '+MetaString+StringOfChar(#0,MetaLength-Length(MetaString));

          // divide the length by 16 and put it in place.
          MetaLength:=MetaLength div 16;
          MetaByte:=MetaLength;
          MetaString[1] := Chr(MetaByte);

          ThreadLog.Log(
            Format('ICY server sending metadata (%s, length %d, length-descriptor %d) to clients',[Copy(MetaString,2,255),Length(MetaString)-1,MetaByte]),
            LL_NORMAL
          );

        end
        else
        begin
          MetaString:=#0;
        end;

      // send the buffer and the metadata
      Connection.WriteBuffer(Buffer.Buffer^,BytesBefore);
      Connection.WriteBuffer(MetaString[1],Length(MetaString));

      // calculate the position for the rest of the data and send it as well.
      if (BytesAfter>0) then
        begin
          RestOfBuffer:=Pointer( Integer(Buffer.Buffer) + BytesBefore );
          Connection.WriteBuffer(RestOfBuffer^,BytesAfter);
        end;

      MetaBytes:=BytesAfter;
    end
    else
    begin
      Connection.WriteBuffer(Buffer.Buffer^,Buffer.Size);
      MetaBytes:=MetaBytes+Buffer.Size;
    end;
  Inc(BytesSent,Buffer.Size);
end;


end.
